/* ========= Utils =========== */
function $(sel, ctx=document){ return ctx.querySelector(sel); }
function $$(sel, ctx=document){ return Array.from(ctx.querySelectorAll(sel)); }
const formatPEN = (n)=> `S/ ${Number(n).toLocaleString('es-PE')}`;

// Fuente única de datos (viene de data.products.js)
const products = (window.PRODUCTS || []).slice();

/* ========= NAV =========== */
class Navigation{
  constructor(){
    this.navbar = $('.navbar');
    this.hamburger = $('#hamburger');
    this.mobileMenu = $('#mobileMenu');
    this.navLinks = $$('.nav-link');
    this.mobileLinks = $$('.mobile-menu a');
    this.isMenuOpen = false;
    this.init();
  }
  init(){ this.bindEvents(); this.handleScroll(); }
  bindEvents(){
    this.hamburger?.addEventListener('click', ()=> this.toggleMobileMenu());
    this.mobileLinks.forEach(l=> l.addEventListener('click', ()=> this.closeMobileMenu()));
    [...this.navLinks, ...this.mobileLinks].forEach(link=>{
      link.addEventListener('click', (e)=> this.handleNavClick(e));
    });
    window.addEventListener('scroll', ()=> this.handleScroll());
    window.addEventListener('resize', ()=>{
      if(window.innerWidth>=768 && this.isMenuOpen) this.closeMobileMenu();
    });
    document.addEventListener('click',(e)=>{
      if(this.isMenuOpen && !this.hamburger?.contains(e.target) && !this.mobileMenu?.contains(e.target)){
        this.closeMobileMenu();
      }
    });
  }
  toggleMobileMenu(){
    this.isMenuOpen = !this.isMenuOpen;
    this.hamburger?.classList.toggle('open', this.isMenuOpen);
    this.mobileMenu?.classList.toggle('open', this.isMenuOpen);
    document.body.style.overflow = this.isMenuOpen ? 'hidden' : '';
  }
  closeMobileMenu(){
    this.isMenuOpen=false;
    this.hamburger?.classList.remove('open');
    this.mobileMenu?.classList.remove('open');
    document.body.style.overflow='';
  }
  handleNavClick(e){
    const href = e.target.getAttribute('href');
    if(href && href.startsWith('#')){
      e.preventDefault();
      const t = document.querySelector(href);
      if(t){ const top = t.offsetTop - 80; window.scrollTo({top, behavior:'smooth'}); }
      this.updateActiveState(href);
    }
  }
  handleScroll(){
    const scrolled = window.scrollY > 50;
    this.navbar?.classList.toggle('scrolled', scrolled);
    this.updateActiveStateOnScroll();
  }
  updateActiveState(href){
    [...this.navLinks, ...this.mobileLinks].forEach(l=>{
      l.classList.toggle('active', l.getAttribute('href')===href);
    });
  }
  updateActiveStateOnScroll(){
    const secs = $$('section[id]');
    const y = window.scrollY + 100;
    secs.forEach(s=>{
      const top=s.offsetTop, bottom=top+s.offsetHeight, id='#'+s.getAttribute('id');
      if(y>=top && y<=bottom) this.updateActiveState(id);
    });
  }
}

/* ========= CARD EFFECTS =========== */
class CardEffects{
  constructor(){
    this.cards = $$('.info-col');
    this.cards.forEach(c=>{
      c.addEventListener('mouseenter',()=>this.t(c,true));
      c.addEventListener('mouseleave',()=>this.t(c,false));
      c.addEventListener('focus',()=>this.t(c,true));
      c.addEventListener('blur',()=>this.t(c,false));
    });
  }
  t(c,on){ requestAnimationFrame(()=>c.classList.toggle('hovered',on)); }
}

/* ========= SCROLL REVEAL =========== */
class ScrollReveal{
  constructor(){
    this.elements = $$('.product-card, .car-card, .brand-strip > div, .brand-card, .img-only-card, .catalog-card');
    this.observer = new IntersectionObserver((es)=>{
      es.forEach(e=>{
        if(e.isIntersecting){
          const delay=[...this.elements].indexOf(e.target)*100;
          setTimeout(()=> e.target.classList.add('fade-up'), delay);
          this.observer.unobserve(e.target);
        }
      });
    },{threshold:.15, rootMargin:'0px 0px -50px 0px'});
    this.elements.forEach(el=>{
      el.classList.add('fade-hidden');
      this.observer.observe(el);
    });
  }
}

/* ========= HERO CAROUSEL =========== */
class EnhancedCarousel{
  constructor(){
    this.currentSlide=0;
    this.slides = $$('.hero-img');
    this.indicators = $$('.slide-indicator');
    this.nextBtn = $('.hero-next');
    this.prevBtn = $('.hero-prev');
    this.hero = $('.hero');
    this.progressBar = $('.progress-bar');
    this.autoTimer=null; this.progressTimer=null; this.isTransitioning=false;
    this.autoplayDuration=7000; this.touchStartX=0; this.touchStartY=0; this.minSwipeDistance=50;
    this.init();
  }
  init(){ if(!this.slides.length) return; this.bind(); this.show(0); this.start(); }
  bind(){
    this.nextBtn?.addEventListener('click',()=>this.next());
    this.prevBtn?.addEventListener('click',()=>this.prev());
    this.indicators.forEach((d,i)=>d.addEventListener('click',()=>this.go(i)));
    this.hero?.addEventListener('mouseenter',()=>this.pause());
    this.hero?.addEventListener('mouseleave',()=>this.resume());
    this.hero?.addEventListener('touchstart',e=>{
      this.touchStartX=e.changedTouches[0].clientX; this.touchStartY=e.changedTouches[0].clientY;
    },{passive:true});
    this.hero?.addEventListener('touchend',e=>{
      const dx=e.changedTouches[0].clientX-this.touchStartX, dy=e.changedTouches[0].clientY-this.touchStartY;
      if(Math.abs(dx)>Math.abs(dy)&&Math.abs(dx)>this.minSwipeDistance){dx>0?this.prev():this.next();}
    },{passive:true});
    document.addEventListener('keydown',e=>{
      if(['INPUT','TEXTAREA'].includes(e.target.tagName)) return;
      if(e.key==='ArrowRight'||e.key===' '){ e.preventDefault(); this.next(); }
      if(e.key==='ArrowLeft'){ e.preventDefault(); this.prev(); }
    });
    document.addEventListener('visibilitychange',()=>document.hidden?this.pause():this.resume());
  }
  show(i,dir='next'){
    if(this.isTransitioning||i===this.currentSlide) return;
    this.isTransitioning=true;
    const prev=this.currentSlide; this.currentSlide=i;
    this.slides.forEach((s,idx)=>{
      s.classList.remove('active','next','prev');
      if(idx===i) s.classList.add('active');
      else if(idx===prev) s.classList.add(dir==='next'?'prev':'next');
    });
    this.indicators.forEach((d,idx)=>d.classList.toggle('active',idx===i));
    setTimeout(()=>this.isTransitioning=false,1000);
    this.resetProgress();
  }
  next(){ this.show((this.currentSlide+1)%this.slides.length,'next'); }
  prev(){ this.show((this.currentSlide-1+this.slides.length)%this.slides.length,'prev'); }
  go(i){ this.show(i,i>this.currentSlide?'next':'prev'); }
  start(){ if(this.slides.length<=1) return; this.stop(); this.autoTimer=setInterval(()=>this.next(),this.autoplayDuration); this.startProgress(); }
  stop(){ if(this.autoTimer){clearInterval(this.autoTimer); this.autoTimer=null;} this.stopProgress(); }
  pause(){ this.stop(); }
  resume(){ if(!document.hidden) this.start(); }
  startProgress(){
    this.stopProgress();
    let p=0,inc=100/(this.autoplayDuration/50);
    this.progressTimer=setInterval(()=>{
      p+=inc; if(this.progressBar) this.progressBar.style.width=`${Math.min(p,100)}%`;
    },50);
  }
  stopProgress(){ if(this.progressTimer){clearInterval(this.progressTimer); this.progressTimer=null;} }
  resetProgress(){ if(this.progressBar) this.progressBar.style.width='0%'; if(this.autoTimer) this.startProgress(); }
}

/* ========= CARRITO =========== */
class ShoppingCart{
  constructor(){ this.items = JSON.parse(localStorage.getItem('cart')) || []; this.init(); }
  init(){ this.bindEvents(); this.updateCartUI(); this.loadProducts(); }
  bindEvents(){
    $('#cartBtn')?.addEventListener('click',()=>this.toggleCart());
    $('#cartBtnMobile')?.addEventListener('click',()=>this.toggleCart());
    $('#closeCart')?.addEventListener('click',()=>this.closeCart());
    $('#cartOverlay')?.addEventListener('click',()=>this.closeCart());
    $('#checkoutBtn')?.addEventListener('click',()=>this.checkout());
    $('#closeModal')?.addEventListener('click',()=>this.closeModal());
    $('#addToCartBtn')?.addEventListener('click',()=>this.addToCartFromModal());
    $('#decreaseQty')?.addEventListener('click',()=>this.changeModalQuantity(-1));
    $('#increaseQty')?.addEventListener('click',()=>this.changeModalQuantity(1));
    $('#modalQuantity')?.addEventListener('change',(e)=>{
      e.target.value = Math.max(1, parseInt(e.target.value) || 1);
    });
  }

  // Grid de HOME con textos (opcional). Si no existe #productsGrid, no hace nada.
  loadProducts(){
    const grid = $('#productsGrid'); if(!grid) return;
    grid.innerHTML = products.map(p=>`
      <div class="product-card rounded-xl overflow-hidden shadow-lg" data-product-id="${p.id}">
        <div class="h-48 bg-white grid place-items-center">
          <img src="${p.image}" alt="${p.name}" class="max-h-48 object-contain p-6" loading="lazy">
        </div>
        <div class="p-6">
          <h3 class="text-xl font-bold mb-2 text-gray-800">${p.name}</h3>
          <p class="text-gray-600 mb-4 text-sm">${p.description || p.short || ''}</p>
          <div class="flex items-center justify-between">
            <span class="text-2xl font-bold text-red-600">${formatPEN(p.price)}</span>
            <button class="add-to-cart-btn bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg" data-product-id="${p.id}">Agregar</button>
          </div>
        </div>
      </div>`).join('');

    $$('.product-card').forEach(card=>{
      card.addEventListener('click',(e)=>{
        if(!e.target.classList.contains('add-to-cart-btn')){
          this.showProductModal(parseInt(card.dataset.productId,10));
        }
      });
    });
    $$('.add-to-cart-btn').forEach(btn=>{
      btn.addEventListener('click',(e)=>{
        e.stopPropagation();
        this.addToCart(parseInt(btn.dataset.productId,10),1);
      });
    });
  }

  showProductModal(id){
    const p = products.find(x=>x.id===id); if(!p) return;
    $('#modalTitle')?.textContent = p.name;
    $('#modalDescription')?.textContent = p.description || p.short || '';
    $('#modalPrice')?.textContent = formatPEN(p.price);
    const imgWrap = $('#modalImage');
    if(imgWrap) imgWrap.innerHTML = `<img src="${p.image}" alt="${p.name}" class="max-h-64 object-contain p-4">`;

    const specs = $('#modalSpecs');
    if(specs){
      specs.innerHTML = p.specs ? Object.entries(p.specs).map(([k,v])=>`
        <div class="flex justify-between py-1 border-b border-gray-100">
          <span class="font-medium text-gray-700">${k}:</span><span class="text-gray-600">${v}</span>
        </div>`).join('') : '';
    }
    const qty = $('#modalQuantity'); if(qty) qty.value = 1;
    const add = $('#addToCartBtn'); if(add) add.dataset.productId = id;

    $('#productModal')?.classList.remove('hidden');
    document.body.style.overflow = 'hidden';
  }
  closeModal(){ $('#productModal')?.classList.add('hidden'); document.body.style.overflow=''; }
  changeModalQuantity(delta){ const input=$('#modalQuantity'); if(input) input.value = Math.max(1, (parseInt(input.value)||1)+delta); }
  addToCartFromModal(){
    const id=parseInt($('#addToCartBtn')?.dataset.productId || '0',10);
    const qty=parseInt($('#modalQuantity')?.value || '1',10);
    this.addToCart(id, qty);
    this.closeModal();
  }

  addToCart(id, qty=1){
    const p=products.find(x=>x.id===id); if(!p) return;
    const ex=this.items.find(i=>i.id===id);
    if(ex) ex.quantity+=qty; else this.items.push({...p, quantity:qty});
    this.save(); this.updateCartUI(); this.bump();
  }
  removeFromCart(id){ this.items = this.items.filter(i=>i.id!==id); this.save(); this.updateCartUI(); }
  updateQuantity(id, qty){
    const it=this.items.find(i=>i.id===id);
    if(it){ if(qty<=0) this.removeFromCart(id); else { it.quantity=qty; this.save(); this.updateCartUI(); } }
  }
  save(){ localStorage.setItem('cart', JSON.stringify(this.items)); }

  updateCartUI(){
    const totalItems = this.items.reduce((s,i)=>s+i.quantity,0);
    const total = this.items.reduce((s,i)=>s+i.price*i.quantity,0);

    [$('#cartCount'), $('#cartCountMobile')].forEach(el=>{
      if(!el) return;
      el.textContent = totalItems;
      el.classList.toggle('hidden', totalItems===0);
      if(totalItems>0){ el.classList.add('cart-badge'); setTimeout(()=>el.classList.remove('cart-badge'),500); }
    });

    const list = $('#cartItems'); const empty = $('#emptyCart'); const footer = $('#cartFooter');
    if(!list) return;

    if(this.items.length===0){
      empty?.classList.remove('hidden');
      footer?.classList.add('hidden');
      list.innerHTML='';
      if(empty) list.appendChild(empty);
    } else {
      empty?.classList.add('hidden');
      footer?.classList.remove('hidden');
      list.innerHTML = this.items.map(item=>`
        <div class="flex items-center space-x-4 p-4 border border-gray-200 rounded-lg mb-4">
          <div class="w-16 h-16 bg-white rounded-lg grid place-items-center">
            <img src="${item.image}" alt="${item.name}" class="max-h-16 object-contain p-2">
          </div>
          <div class="flex-1">
            <h4 class="font-bold text-gray-800 text-sm">${item.name}</h4>
            <p class="text-red-600 font-bold">${formatPEN(item.price)}</p>
            <div class="flex items-center space-x-2 mt-2">
              <button class="quantity-btn w-8 h-8 border border-gray-300 rounded" onclick="cart.updateQuantity(${item.id}, ${item.quantity-1})" aria-label="Quitar uno">-</button>
              <span class="w-8 text-center text-sm">${item.quantity}</span>
              <button class="quantity-btn w-8 h-8 border border-gray-300 rounded" onclick="cart.updateQuantity(${item.id}, ${item.quantity+1})" aria-label="Agregar uno">+</button>
              <button class="ml-2 text-red-500 hover:text-red-700" onclick="cart.removeFromCart(${item.id})" aria-label="Eliminar del carrito">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
              </button>
            </div>
          </div>
        </div>`).join('');
      $('#cartTotal')?.textContent = formatPEN(total);
    }
  }
  toggleCart(){
    const s=$('#cartSidebar'); const o=$('#cartOverlay');
    s?.classList.toggle('open'); o?.classList.toggle('hidden');
    document.body.style.overflow = s?.classList.contains('open') ? 'hidden' : '';
  }
  closeCart(){ $('#cartSidebar')?.classList.remove('open'); $('#cartOverlay')?.classList.add('hidden'); document.body.style.overflow=''; }
  bump(){ const b=$('#cartBtn'); if(b){ b.style.transform='scale(1.1)'; setTimeout(()=>b.style.transform='scale(1)',200);} }

  checkout(){
    if(this.items.length===0) return;
    const total = this.items.reduce((s,i)=>s+i.price*i.quantity,0);
    let msg = "🛒 *PEDIDO DE BATERÍAS* 🔋\n\n📋 *PRODUCTOS:*\n";
    this.items.forEach((it,idx)=>{
      msg += `${idx+1}. ${it.name}\n   💰 Precio: ${formatPEN(it.price)}\n   📦 Cantidad: ${it.quantity}\n   💵 Subtotal: ${formatPEN(it.price*it.quantity)}\n\n`;
    });
    msg += `💰 *TOTAL: ${formatPEN(total)}*\n\n📍 Por favor confirmen disponibilidad y tiempo de entrega.\n🚚 ¿Incluye instalación gratuita?\n\n¡Gracias! 😊`;
    window.open(`https://wa.me/5255123456789?text=${encodeURIComponent(msg)}`,'_blank','noopener,noreferrer');
  }
}

/* ========= HOME: SOLO IMÁGENES DE MARCAS (fallback si no usas brands.js) =========== */
function renderHomeBrandsFromProducts(){
  const grid = document.getElementById('productsBrandsGrid');
  if (!grid) return;
  // Si brands.js ya pintó, no duplicamos
  if (grid.childElementCount > 0) return;

  const seen = new Set();
  const tiles = [];
  for (const p of products){
    if (!p.brand || seen.has(p.brand)) continue;
    seen.add(p.brand);
    tiles.push(p);
  }

  grid.innerHTML = tiles.map(p => `
    <a href="productos.html?brand=${encodeURIComponent(p.brand)}"
       class="img-only-card block bg-white border rounded-xl overflow-hidden shadow-sm hover:shadow-md transition"
       aria-label="${p.brand}">
      <div class="aspect-square grid place-items-center p-4">
        <img src="${p.image}" alt="${p.name}" loading="lazy" class="max-w-[85%] max-h-[85%] object-contain">
      </div>
      <span class="sr-only">${p.name}</span>
    </a>
  `).join('');
}

/* ========= CATÁLOGO (productos.html) =========== */
function qsParam(name){ return new URLSearchParams(location.search).get(name) || ''; }
function uniqBrands(){ return [...new Set(products.map(p=>p.brand).filter(Boolean))]; }

function renderCatalogPage(){
  const holder = document.getElementById('catalogPage') || document.getElementById('allProductsGrid')?.closest('main'); // compat
  const gridCompat = document.getElementById('allProductsGrid');
  const titleCompat = document.getElementById('productsPageTitle');

  // Si usas el layout simple (productos.html que te pasé), renderizamos directo al #allProductsGrid
  if (gridCompat && titleCompat){
    const brand = qsParam('brand').toLowerCase();
    let data = products.slice();
    if(brand) data = data.filter(p=>p.brand===brand);

    // título
    if(brand && window.BRANDS){
      const found = window.BRANDS.find(b=>b.key===brand);
      titleCompat.textContent = found ? `Productos — ${found.name}` : 'Productos';
    } else {
      titleCompat.textContent = 'Todos los productos';
    }

    gridCompat.innerHTML = data.map(p=>`
      <div class="catalog-card rounded-xl overflow-hidden bg-white shadow">
        <div class="h-44 bg-white grid place-items-center">
          <img src="${p.image}" alt="${p.name}" class="max-h-40 object-contain p-4" loading="lazy">
        </div>
        <div class="p-5">
          <h3 class="text-base font-bold mb-2 text-gray-800 line-clamp-2">${p.name}</h3>
          <p class="text-sm text-gray-500 mb-4 line-clamp-2">${p.short || ''}</p>
          <div class="flex items-center justify-between">
            <span class="text-xl font-extrabold text-red-600">${formatPEN(p.price)}</span>
            <button class="add-to-cart-btn bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg" data-product-id="${p.id}">
              Agregar
            </button>
          </div>
        </div>
      </div>
    `).join('');

    gridCompat.querySelectorAll('.add-to-cart-btn').forEach(btn=>{
      btn.addEventListener('click',(e)=>{
        e.stopPropagation();
        const id=parseInt(btn.dataset.productId,10);
        if(window.cart) cart.addToCart(id,1);
      });
    });
    gridCompat.querySelectorAll('.catalog-card').forEach(card=>{
      card.addEventListener('click',(e)=>{
        if(e.target.classList.contains('add-to-cart-btn')) return;
        const id = parseInt(card.querySelector('.add-to-cart-btn')?.dataset.productId || '0',10);
        if(window.cart && typeof cart.showProductModal === 'function'){ cart.showProductModal(id); }
      });
    });
    return;
  }

  // Si usas un layout con contenedor #catalogPage (más avanzado)
  if(!holder) return;

  const activeBrand = qsParam('brand').toLowerCase();
  const brands = uniqBrands();

  const pills = ['todos', ...brands].map(b=>{
    const active = (b==='todos' && !activeBrand) || (b!=='todos' && b===activeBrand);
    return `<button class="filter-pill ${active?'active':''}" data-brand="${b==='todos'?'':b}">${b==='todos'?'Todos':b}</button>`;
  }).join('');

  holder.innerHTML = `
    <div class="container mx-auto px-4">
      <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3 mb-6">
        <h1 class="text-3xl md:text-4xl font-extrabold" style="color:var(--ink-900)">Catálogo de Baterías</h1>
        <a href="productos.html" class="btn-dark px-5 py-2 rounded-full font-bold">Ver todos</a>
      </div>
      <div class="flex flex-wrap gap-3 mb-8" id="brandPills">${pills}</div>
      <div id="catalogGrid" class="grid grid-cols-2 md:grid-cols-3 xl:grid-cols-6 gap-4 md:gap-6"></div>
    </div>
  `;

  const renderGrid = ()=>{
    const current = qsParam('brand').toLowerCase();
    const data = current ? products.filter(p=>p.brand===current) : products.slice();
    const grid = document.getElementById('catalogGrid');
    grid.innerHTML = data.map(p => `
      <div class="catalog-card rounded-xl overflow-hidden bg-white shadow" data-product-id="${p.id}">
        <div class="aspect-square bg-white grid place-items-center p-4">
          <img src="${p.image}" alt="${p.name}" class="max-w-[85%] max-h-[85%] object-contain" loading="lazy">
        </div>
        <div class="p-4">
          <h3 class="font-bold text-sm md:text-base mb-1" style="color:var(--ink-900)">${p.name}</h3>
          <div class="flex items-center justify-between">
            <span class="price-tag text-lg md:text-xl">${formatPEN(p.price)}</span>
            <button class="add-to-cart-btn bg-red-600 hover:bg-red-700 text-white px-3 py-1.5 rounded-lg text-sm"
                    data-product-id="${p.id}">
              Agregar
            </button>
          </div>
        </div>
      </div>
    `).join('');

    grid.querySelectorAll('.catalog-card').forEach(card=>{
      card.addEventListener('click', (e)=>{
        if(e.target.classList.contains('add-to-cart-btn')) return;
        const id = parseInt(card.dataset.productId,10);
        if(window.cart && typeof cart.showProductModal === 'function'){ cart.showProductModal(id); }
      });
    });

    grid.querySelectorAll('.add-to-cart-btn').forEach(btn=>{
      btn.addEventListener('click', (e)=>{
        e.stopPropagation();
        const id = parseInt(btn.dataset.productId,10);
        if(window.cart){ cart.addToCart(id,1); }
      });
    });
  };

  const pillsWrap = document.getElementById('brandPills');
  pillsWrap.addEventListener('click',(e)=>{
    const btn = e.target.closest('.filter-pill'); if(!btn) return;
    const brand = btn.dataset.brand || '';
    const url = new URL(location.href);
    if(brand) url.searchParams.set('brand', brand); else url.searchParams.delete('brand');
    history.replaceState(null, '', url.toString());
    pillsWrap.querySelectorAll('.filter-pill').forEach(el=>el.classList.remove('active'));
    btn.classList.add('active');
    renderGrid();
  });

  renderGrid();
}

/* ========= FABs / PANELS =========== */
function openWidget(type){
  const wa = $('#panelWhatsApp');
  const call = $('#panelCall');
  if(type === 'whatsapp'){ wa?.classList.add('open'); call?.classList.remove('open'); }
  if(type === 'call'){ call?.classList.add('open'); wa?.classList.remove('open'); }
}
function closeWidget(id){ const el = document.getElementById(id); el?.classList.remove('open'); }
function bindWidgets(){
  const fabWA = $('#fabWhatsApp');
  const fabCall = $('#fabCall');
  const closeBtns = $$('.widget-close');

  fabWA?.addEventListener('click', ()=> openWidget('whatsapp'));
  fabCall?.addEventListener('click', ()=> openWidget('call'));
  closeBtns.forEach(btn=> btn.addEventListener('click', ()=> closeWidget(btn.getAttribute('data-close'))));

  document.addEventListener('click', (e)=>{
    const p1 = $('#panelWhatsApp'); const p2 = $('#panelCall');
    const inside = p1?.contains(e.target) || p2?.contains(e.target);
    const isFab = e.target.closest?.('.fab-btn');
    if(!inside && !isFab){ p1?.classList.remove('open'); p2?.classList.remove('open'); }
  });
  document.addEventListener('keydown', (e)=>{
    if(e.key === 'Escape'){ $('#panelWhatsApp')?.classList.remove('open'); $('#panelCall')?.classList.remove('open'); }
  });

  // Quick WhatsApp
  const quick = $('#waQuickMsg'); const send = $('#waSendToCarlos');
  send?.addEventListener('click', ()=>{
    const txt = (quick?.value || '').trim();
    const url = 'https://wa.me/5255111111111?text=' + encodeURIComponent(txt || 'Hola, necesito asesoría 🙌');
    window.open(url, '_blank', 'noopener,noreferrer');
  });

  // Botones con data-open
  $$('[data-open="whatsapp"]').forEach(el=> el.addEventListener('click', (ev)=>{ ev.preventDefault(); openWidget('whatsapp'); }));
  $$('[data-open="call"]').forEach(el=> el.addEventListener('click', (ev)=>{ ev.preventDefault(); openWidget('call'); }));
}

/* ========= MAPA ========= */
function initMap(){
  const el = document.getElementById('map'); if(!el) return;
  const location = { lat: 19.4326, lng: -99.1332 };
  const map = new google.maps.Map(el, {
    zoom: 15, center: location,
    styles:[
      {"featureType":"all","elementType":"geometry.fill","stylers":[{"weight":"2.00"}]},
      {"featureType":"all","elementType":"geometry.stroke","stylers":[{"color":"#9c9c9c"}]},
      {"featureType":"all","elementType":"labels.text","stylers":[{"visibility":"on"}]},
      {"featureType":"landscape","elementType":"all","stylers":[{"color":"#f2f2f2"}]},
      {"featureType":"landscape","elementType":"geometry.fill","stylers":[{"color":"#ffffff"}]},
      {"featureType":"poi","elementType":"all","stylers":[{"visibility":"off"}]},
      {"featureType":"road","elementType":"all","stylers":[{"saturation":-100},{"lightness":45}]},
      {"featureType":"road.highway","elementType":"all","stylers":[{"visibility":"simplified"}]},
      {"featureType":"transit","elementType":"all","stylers":[{"visibility":"off"}]},
      {"featureType":"water","elementType":"geometry.fill","stylers":[{"color":"#c8d7d4"}]}
    ]
  });
  const marker = new google.maps.Marker({
    position: location, map, title: 'CambiaTuBateria - Nuestra Ubicación',
    animation: google.maps.Animation.DROP,
    icon: {
      url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
        <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
          <circle cx="20" cy="20" r="18" fill="#DC2626" stroke="#ffffff" stroke-width="4"/>
          <circle cx="20" cy="20" r="8" fill="#ffffff"/>
        </svg>`),
      scaledSize: new google.maps.Size(40, 40), anchor: new google.maps.Point(20, 20)
    }
  });
  const infoWindow = new google.maps.InfoWindow({
    content: `
      <div class="p-2">
        <h3 class="font-bold text-lg mb-1 text-gray-800">CambiaTuBateria</h3>
        <p class="text-gray-600 mb-1">Av. Principal 123<br/>Ciudad de México, CP 01234</p>
        <p class="text-gray-600 mb-2">📞 +52 55 1234 5678</p>
        <a href="https://www.google.com/maps/dir/?api=1&destination=${location.lat},${location.lng}" target="_blank" class="inline-block bg-slate-600 hover:bg-slate-700 text-white px-3 py-2 rounded-lg text-sm">Cómo llegar</a>
      </div>`
  });
  marker.addListener('click', ()=> infoWindow.open(map, marker));
  setTimeout(()=> infoWindow.open(map, marker), 1000);

  // Fallback
  setTimeout(()=>{
    if(!el.hasChildNodes()){
      el.innerHTML = `<div class="flex items-center justify-center h-full bg-gray-100 text-gray-500">
        <div class="text-center">
          <p class="text-lg font-medium">Mapa no disponible</p>
          <p class="text-sm">Av. Principal 123, Ciudad de México</p>
          <a href="https://www.google.com/maps/search/Av.+Principal+123+Ciudad+de+México" target="_blank" class="inline-block mt-3 bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm">Ver en Google Maps</a>
        </div>
      </div>`;
    }
  }, 3000);
}
window.initMap = initMap;

/* ========= INIT ========= */
let cart;
document.addEventListener('sections:loaded', ()=>{
  new Navigation();
  new EnhancedCarousel();
  new CardEffects();
  new ScrollReveal();
  bindWidgets();
  cart = new ShoppingCart();

  // Home: solo imágenes de marcas (si no se usó brands.js)
  renderHomeBrandsFromProducts();

  // Catálogo: productos.html
  renderCatalogPage();

  const y = document.getElementById('y'); if(y) y.textContent = new Date().getFullYear();
});

